package com.eis.tcp;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.LinkedList;
import java.util.List;

import javax.resource.ResourceException;
import javax.resource.spi.ActivationSpec;
import javax.resource.spi.BootstrapContext;
import javax.resource.spi.ResourceAdapter;
import javax.resource.spi.ResourceAdapterInternalException;
import javax.resource.spi.endpoint.MessageEndpointFactory;
import javax.resource.spi.work.Work;
import javax.resource.spi.work.WorkException;
import javax.resource.spi.work.WorkManager;
import javax.transaction.xa.XAResource;

/**
 * Inbound ResourceAdapter Komponente.
 * 
 * Ist eine JavaBean (implementiert deshalb Serializable und
 * muss einen parameterlosen Konstruktor haben),
 * die config-property Einträge des Deployment Descriptors
 * werden entsprechend an die RA Instanz übergeben.
 *
 * Diese Beispielimplementierung betrachtet Telnet-Clients als EIS
 * und implementiert ein triviales Kommunikationsprotokoll.
 * 
 * Nach 16.4 der Spec ist eine Implementierung für hashCode
 * und equals auf der Basis der Klasseneigenschaften zu liefern.
 */
public final class TCPResourceAdapter implements ResourceAdapter, Serializable {

	private static final long serialVersionUID = -3660518154888161024L;

	private Work serverWork = null;
	
	/**
	 * Der BootstrapContext kann als Member in der RA Instanz vorgehalten
	 * werden - der Context muss für während der gesamten Lebensspanne des
	 * RA intakt bleiben.
	 */
	private BootstrapContext bootstrapContext = null;
	
	/**
	 * Die config-property Einträge des DD werden hier eingetragen -
	 * hier ist Beispielhaft der Port für den ServerSocket konfiguriert.
	 */
	private int port = 1234;

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	/**
	 * Die Liste der ActivationSpec wird benötigt, um
	 * alle Endpoints mit Nachrichten beliefern zu können.
	 */
	private final List<TCPActivationSpec> activationSpecList = 
		new LinkedList<TCPActivationSpec>();
	
	/**
	 * Zugriff auf des Servers WorkManager Instanz.
	 * 
	 * @return Der mit dem BootstrapContext assoziierte WorkManager
	 */
	WorkManager getWorkManager() {
		return this.bootstrapContext.getWorkManager();
	}

	/**
	 * Der Server meldet dem Resource Adapter einen weiteren Endpunkt.
	 * 
	 * Das passiert faktisch immer dann, wenn eine MDB deployed wird, die
	 * den messaging-type dieses Resource Adapters implementiert.
	 * Die MessageEndpointFactory Instanz wird vom Server bereitgestelllt.
	 * 
	 * An dieser Stelle kann noch kein MessageEndpoint generiert werden
	 * (Vermutung: der generierte Endpoint Proxy prüft die Existenz des
	 * Endpoints, der ja erst NACH passieren dieser Methode zur Verfügung
	 * steht). Deshalb wird die ActivationSpec mit der MessageEndpointFactory
	 * assoziiert.
	 */
	public void endpointActivation(final MessageEndpointFactory mef,
			final ActivationSpec activationSpec) throws ResourceException {
		if (!(activationSpec instanceof TCPActivationSpec)) {
			throw new ResourceException("invalid activation spec type");
		}
		
		TCPActivationSpec tcpActivationSpec = (TCPActivationSpec) activationSpec;
		tcpActivationSpec.validate(); // kann eine ResourceException werfen
		
		// Der Server liefert einen Proxy, der gleichzeitig
		// MessageEndpoint und TCPMessageListener ist
		// Statt null kann auch eine passende XAResource übergeben werden
		try {
			Method endpointMethod =
				TCPMessageListener.class.getMethod("onMessage", new Class[] { String.class });
			tcpActivationSpec.setMessageEndpoint(mef, endpointMethod);
		} catch (SecurityException e) {
			throw new ResourceAdapterInternalException(e);
		} catch (NoSuchMethodException e) {
			throw new ResourceAdapterInternalException(e);
		}
		synchronized(this.activationSpecList) {
			this.activationSpecList.add(tcpActivationSpec);
		}
	}

	/**
	 * Der Server deaktiviert einen Endpunkt.
	 * 
	 * Faktisch wurde eine MDB undeployed.
	 */
	public void endpointDeactivation(MessageEndpointFactory mef,
			ActivationSpec activationSpec) {
		synchronized(this.activationSpecList) {
			this.activationSpecList.remove(activationSpec);
		}
	}

	/**
	 * Teil des message inflow contracts: gibt ein Array von XAResource
	 * Objekten, die anhand des Arrays von ActivationSpec Objekten erstellt
	 * wird zurück, wird im Falle eines Systemabsturzes gerufen.
	 * 
	 * @return Array von XAResource Objekten, null: transaction inflow wird
	 * nicht unterstützt
	 */
	public XAResource[] getXAResources(ActivationSpec[] activationSpecs)
			throws ResourceException {
		return null;
	}

	/**
	 * Benachrichtigt alle angemeldeten MessageEndpoints
	 * mit der gegebenen Nachricht.
	 * 
	 * Wir müssen an der factoryMap synchronisieren, damit nicht
	 * deaktivierte Endpoints benachrichtigt werden. Da nur Nachrichten
	 * zugestellt werden sollen wird der betroffene Block aber schnell
	 * durchlaufen.
	 *  
	 * @param message Die Nachricht
	 */
	void sendMessage(String message) {
		synchronized(this.activationSpecList) {
			for (TCPActivationSpec activationSpec : this.activationSpecList) {
				activationSpec.sendMessage(message);
			}
		}
	}

	/**
	 * Der Resource Adapter wird vom Server gestartet.
	 * 
	 * Der BootstrapContext wird gesichert und der Service Thread dieses
	 * RA wird gestartet. Der Service Thread repräsentiert einen
	 * ServerSocket (so wie in diesem Beispiel), ein DatagramSocket 
	 * oder einen Polling Thread. In der Praxis sind diese Details von
	 * einer Java API gekapselt. 
	 */
	public void start(final BootstrapContext bootstrapContext)
			throws ResourceAdapterInternalException {

		/*
		 * Laut Spek muss die BootstrapContext Instanz 
		 * während der geamten Lebenszeit des ResourceAdapters
		 * intakt beleiben:
		 */
		this.bootstrapContext = bootstrapContext;

		/*
		 * Die eigentliche Arbeit darf aber nicht in diesem Thread durchgeführt
		 * werden. Man muss den WorkManager des Servers nutzen.
		 * 
		 * Hier wird ein einfacher Worker gestartet, möglich sind viele andere
		 * Optionen (siehe WorkerManager Interface)
		 */
		final WorkManager workManager = bootstrapContext.getWorkManager();
		
		try {
			// Die Klasse ServerWork repräsentiert den Service einer
			// Java API und wird hier gestartet:
			this.serverWork = new ServerWork(this);
			try {
				workManager.startWork(this.serverWork);
			} catch (WorkException workException) {
				throw new ResourceAdapterInternalException(workException);
			}
		} catch (IOException ioException) {
			throw new ResourceAdapterInternalException(ioException);
		}
	}

	/**
	 * Der Resource Adapter wird vom Server gestopt.
	 * 
	 * Alle Ressourcen werden hier wieder freigegeben.
	 */
	public void stop() {
		if (this.serverWork != null) {
			this.serverWork.release();
		}
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + port;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final TCPResourceAdapter other = (TCPResourceAdapter) obj;
		if (port != other.port)
			return false;
		return true;
	}
}