package com.eis.tcp;

import java.io.Serializable;
import java.lang.reflect.Method;

import javax.resource.ResourceException;
import javax.resource.spi.ActivationSpec;
import javax.resource.spi.InvalidPropertyException;
import javax.resource.spi.ResourceAdapter;
import javax.resource.spi.UnavailableException;
import javax.resource.spi.endpoint.MessageEndpoint;
import javax.resource.spi.endpoint.MessageEndpointFactory;

/**
 * Mit einer AktivationSpec Instanz übergibt ein Message Endpoint seine
 * Konfigurationsparameter bei der Endpoint Registrierung.
 *  
 * Die Implementierung ist JavaBean konform und der Resouerce Adapter DD kann  
 * mit dem required-config-property Element das Setzen entsprechender 
 * Konfigurationsparameter erzwingen - das ist der Weg, mit dem Endpoint 
 * Provider Informationen an den Resource Adapter weiterreichen.
 * 
 * Ein Resource Adapter kann auch mehrere ActivationSpec Implementierungen für
 * verscheidene Message Listener Typen bereitstellen.
 * 
 * In Section 16.4 der Spec ist gefordert, dass die Standard equals Implementierung
 * nicht überschrieben wird (no two Java objects are considerd equals). 
 */
public final class TCPActivationSpec implements ActivationSpec, Serializable {
	
	private static final long serialVersionUID = -6475040405827259828L;

	private Method endpointMethod = null;
	
	private ResourceAdapter resourceAdapter = null;
	
	private MessageEndpointFactory mef = null;
	
	public ResourceAdapter getResourceAdapter() {
		return this.resourceAdapter;
	}

	public void setResourceAdapter(ResourceAdapter resourceAdapter)
			throws ResourceException {
		this.resourceAdapter = resourceAdapter;
	}

	/**
	 * Kann im Rahmen der Endpoint Aktivierung gerufen werden.
	 * 
	 * Hier kann noch ein letztes Mal der Inhalt der ActivationSpec
	 * Parameter geprüft werden. Jedoch: nicht der Server ruft diese Methode,
	 * sonder sie ist vom Resource Adapter zu rufen.
	 */
	public void validate() throws InvalidPropertyException {
		// hier können zum Beispiel die vom Endpoint Provider übergebenen
		// Parameter geprüft werden..
		if (getMessage() == null || getMessage().trim().length() == 0) {
			throw new InvalidPropertyException("Endpoint Provider must provide a non-empty message!");
		}
	}
	
	/**
	 * Die Bereitstellung des Parameters "Messsage" wird im DD des RA erzwungen
	 * durch das Element required-config-property.
	 */
	private String message = null;

	public String getMessage() {
		return message;
	}

	public void setMessage(String message) {
		this.message = message;
	}

	/**
	 * Wird im Zuge der endpointActivation vom ResourceAdapter gerufen.
	 * 
	 * Hinweis: an dieser Stelle darf noch nicht mef.createEndpoint
	 * gerufen werden, da der Endpoint noch nicht existiert.
	 */
	void setMessageEndpoint(MessageEndpointFactory mef, Method endpointMethod) {
		this.mef = mef;
		this.endpointMethod = endpointMethod;
	}

	/**
	 * In der ActivationSpec sind alle Resourcen für
	 * das Zustellen einer Nachricht gebündelt.
	 * 
	 * @param message Die zu übertragende Message
	 */
	void sendMessage(String message) {
		try {
			MessageEndpoint endpoint = (MessageEndpoint) mef.createEndpoint(null);
			try {
				// signalisiert den Beginn einer Transaktion
				endpoint.beforeDelivery(this.endpointMethod);
				// versucht die Message zuzustellen, dabei können auch
				// Informationen der ActivationSpec verwendet werden
				((TCPMessageListener) endpoint).onMessage(getMessage() + message);
			} catch (NoSuchMethodException e) {
				// .. individuelle Fehlerbehandlung 
			} catch (ResourceException e) {
				// .. individuelle Fehlerbehandlung 
			} finally {
				if (endpoint != null) {
					try {
						// signalisiert das Ende einer Transaktion
						endpoint.afterDelivery();
					} catch (ResourceException e) {
						// .. individuelle Fehlerbehandlung 
					}
					// der endpoint muss unbedingt wieder freigegeben werden
					endpoint.release();
				}
			}
		} catch (UnavailableException e) {
			// .. individuelle Fehlerbehandlung 
		}
	}
}