package com.eis.tcp;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;

import javax.resource.spi.work.Work;
import javax.resource.spi.work.WorkException;

/**
 * Repräsentiert den eigentlichen Service Thread.
 */
final class ServerWork implements Work {

	private final ServerSocket service;
	
	private final TCPResourceAdapter resourceAdapter;

	/**
	 * Der konstruktor ist der korrekte Ort für das Binden aller Ressourcen.
	 * 
	 * @param workManager
	 * @param factoryMap
	 * @param port
	 * 
	 * @throws IOException Wenn der ServerSocket nicht erstellt werden kann...
	 */
	ServerWork(final TCPResourceAdapter resourceAdapter) 
			throws IOException {
		
		this.resourceAdapter = resourceAdapter;
		// Startet einen ServerSocket
		this.service = new ServerSocket(resourceAdapter.getPort());
	}

	public void run() {
		try {
			while (!service.isClosed()) {
				try {
					final Socket socket = service.accept();
					/*
					 * Statt die Arbeit hier direkt zu erledigen, werden
					 * Work-Einheiten über den work manager gestartet. Damit
					 * wird dieser Workerthread nicht zu lange von seiner
					 * eigentlichen Aufgabe (das Lauschen auf Ereignissen)
					 * abgehalten.
					 */
					try {
						if (!service.isClosed() && socket != null && socket.isConnected()) {
							// Wenn ein Request eintrifft, wird sein Inhalt nebenläufig verarbeitet
							Work requestProcessor = new MessageWork(socket, this); 
							resourceAdapter.getWorkManager().scheduleWork(requestProcessor);
						}
					} catch (WorkException e) {
						// well..
					}
				} catch (IOException e) {
					// well..
				} 
			}
		} catch (Exception e) {
			// Fehlerbehandlung für alle Fehler außer WorkException oder 
			// IOException in der Hauptschleife 
		} finally {
			// Auf alle Fälle ServerSocket schließen
			release();
		}
	}

	/**
	 * Freigabe des Worker Threads.
	 * 
	 * Wird gerufen, wenn der RA gestoppt wird {@link TCPResourceAdapter#stop()}
	 * oder wenn der WorkManager entscheidet, diesen Thread zu stoppen.    
	 */
	public void release() {
		if (this.service != null) {
			try {
				// ServerSocket close ist synchronisiert
				this.service.close();
			} catch (IOException e) {
				// well...
			}
		}
	}

	TCPResourceAdapter getResourceAdapter() {
		return this.resourceAdapter;
	}
}