package com.eis.tcp;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import javax.resource.spi.work.Work;

/**
 * Eine Work-Implementierung zur Abarbeitung eines einzelenen EIS Client Requests.
 * 
 * Die Implementierung dient Testzwecken und erlaubt einem Telnet Client
 * eine Verbindung. Das Protokoll der Sitzung: es werden alle Eingaben entgegen
 * genommen, bis ein ENTER kommt. Dann wird die Zeile als Nachricht an alle 
 * registrierten Endpoints geschickt und als Echo an den Telnet-Client gesendet.
 * 
 * Die Verbindung und dieser Worker Thread werden abgebrochen, wenn der Telnet-Client
 * "exit" als Zeile eingibt.
 * 
 * In der Praxis würde in dieser Klasse die korrekte Abarbeitung des EIS-Protokolls
 * sowie die Extrahierung von Nachrichten imlpementiert werden. Normalerweise kommt
 * auch hier die Java API des EIS zum Einsatz.
 */
final class MessageWork implements Work {

	private final ServerWork serverWork;
	
	private final Socket socket;

	MessageWork(final Socket socket, final ServerWork serverWork) {
		this.serverWork = serverWork;
		this.socket = socket;
	}

	public void run() {
		// extrahiere den Inhalt des Requests:
		InputStream inputStream = null;
		OutputStream outputStream = null;
		try {
			// der Client bekommt nun noch eine Antwort:
			inputStream = this.socket.getInputStream();
			outputStream = this.socket.getOutputStream();
			
			// hier nur das Echo der Anfrage und Weitergabe an die MDB
			StringBuilder sb = new StringBuilder();
			int c = 13;
			boolean exit = false;
			while (!exit && (c = inputStream.read()) != -1) {
				// wir interpretieren 13 (ENTER) als Ende der Eingabe
				sb.append((char) c);
				if (c == 13) {
					String message = sb.toString().trim();
					// exit heißt Verbindung abbrechen!
					if ("exit".equals(message)) {
						exit = true;
					} else {
						outputStream.write(("> " + message + "\n\r").getBytes());
						outputStream.flush();
						// Liefere dem ResourceAdapter die Nachricht, er
						// wird alle MessageEndpoints benachtrichtigen
						this.serverWork.getResourceAdapter().sendMessage(message);
						// Platz für die nächste Nachricht
						sb.setLength(0);
					}
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (outputStream != null) {
				try {
					outputStream.close();
				} catch (IOException e) {
					// well...
				}
			}
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (IOException e) {
					// well...
				}
			}
			release();
		}
	}

	public void release() {
		if (this.socket != null) {
			try {
				this.socket.close();
			} catch (IOException e) {
				// well..
			}
		}
	}
}
